import os
import json
import base64
import tkinter as tk
from tkinter import filedialog, simpledialog, messagebox
from cryptography.hazmat.primitives.ciphers.aead import AESGCM
from argon2.low_level import hash_secret_raw, Type

# --- Key Derivation using Argon2id ---
def derive_key(password: str, salt: bytes) -> bytes:
    return hash_secret_raw(
        password.encode(),  # Must be positional
        salt,
        time_cost=3,
        memory_cost=65536,
        parallelism=2,
        hash_len=32,
        type=Type.ID
    )

# --- Encrypt File ---
def encrypt_file(path: str, password: str):
    salt = os.urandom(16)
    nonce = os.urandom(12)
    key = derive_key(password, salt)

    with open(path, 'rb') as f:
        data = f.read()

    aes = AESGCM(key)
    ciphertext = aes.encrypt(nonce, data, None)

    payload = {
        'salt': base64.b64encode(salt).decode(),
        'nonce': base64.b64encode(nonce).decode(),
        'ciphertext': base64.b64encode(ciphertext).decode(),
        'filename': os.path.basename(path)
    }

    output_path = path + ".lock"
    with open(output_path, 'w') as f:
        json.dump(payload, f)

    messagebox.showinfo("Success", f"File encrypted:\n{output_path}")

# --- Decrypt File ---
def decrypt_file(path: str, password: str):
    try:
        with open(path, 'r') as f:
            payload = json.load(f)

        salt = base64.b64decode(payload['salt'])
        nonce = base64.b64decode(payload['nonce'])
        ciphertext = base64.b64decode(payload['ciphertext'])
        filename = payload.get('filename', 'output')

        key = derive_key(password, salt)
        aes = AESGCM(key)
        plaintext = aes.decrypt(nonce, ciphertext, None)

        out_path = os.path.join(os.path.dirname(path), filename)
        with open(out_path, 'wb') as f:
            f.write(plaintext)

        messagebox.showinfo("Success", f"File decrypted:\n{out_path}")
    except Exception as e:
        messagebox.showerror("Error", f"Decryption failed.\nCheck your password.\n\nDetails:\n{str(e)}")

# --- Main GUI Logic ---
def main():
    root = tk.Tk()
    root.withdraw()

    file_path = filedialog.askopenfilename(title="Select a file to encrypt or decrypt")
    if not file_path:
        return

    password = simpledialog.askstring("Password", "Enter password:", show='*')
    if not password:
        return

    if file_path.endswith(".lock"):
        decrypt_file(file_path, password)
    else:
        encrypt_file(file_path, password)

if __name__ == "__main__":
    main()
